/**
 * GrowDiary – Frontend App v1.2
 *
 * Fixes in this version:
 *  1. Edit diary: formToObj() was overwriting diary_id because the hidden
 *     input is inside the form. Now diary_id is set AFTER the spread so it
 *     always wins, and formToObj skips hidden inputs.
 *  2. Height toggle in/cm (DB stores cm, display converts like temp).
 *  3. Photo lightbox on weekly gallery images.
 */
jQuery( function( $ ) {

    const AJAX      = GrowDiary.ajax_url;
    const NONCE     = GrowDiary.nonce;
    const IS_LOGGED = GrowDiary.logged_in === 'yes';

    /* ===================================================================
       UNIT PREFERENCES  (persisted in localStorage)
    =================================================================== */
    let tempUnit   = localStorage.getItem( 'gd_temp_unit' )   || 'F'; // °F default
    let heightUnit = localStorage.getItem( 'gd_height_unit' ) || 'in'; // inches default

    // --- Temperature helpers ---
    function cToF( c ) { return c !== '' && c !== null ? ( parseFloat(c) * 9/5 + 32 ).toFixed(1) : ''; }
    function fToC( f ) { return f !== '' && f !== null ? ( ( parseFloat(f) - 32 ) * 5/9 ).toFixed(2) : ''; }
    function tempToDisplay( c ) {
        if ( c === null || c === '' || isNaN( parseFloat(c) ) ) return '';
        return tempUnit === 'F' ? cToF(c) : parseFloat(c).toFixed(1);
    }
    function displayToC( v ) { return v === '' ? '' : ( tempUnit === 'F' ? fToC(v) : v ); }
    function tempLabel()   { return tempUnit   === 'F'  ? '°F' : '°C'; }

    // --- Height helpers (DB stores cm) ---
    function cmToIn( cm ) { return cm !== '' && cm !== null ? ( parseFloat(cm) / 2.54 ).toFixed(1) : ''; }
    function inToCm( i  ) { return i  !== '' && i  !== null ? ( parseFloat(i) * 2.54 ).toFixed(1)  : ''; }
    function heightToDisplay( cm ) {
        if ( cm === null || cm === '' || isNaN( parseFloat(cm) ) ) return '';
        return heightUnit === 'in' ? cmToIn(cm) : parseFloat(cm).toFixed(1);
    }
    function displayToCm( v ) { return v === '' ? '' : ( heightUnit === 'in' ? inToCm(v) : v ); }
    function heightLabel() { return heightUnit === 'in' ? 'in' : 'cm'; }
    function heightPlaceholder() { return heightUnit === 'in' ? 'e.g. 18' : 'e.g. 45'; }

    // Inject the two toggle buttons into the week modal header (once)
    function injectWeekToggles() {
        if ( $( '#gd-temp-toggle' ).length ) {
            // Already injected – just update labels
            $( '#gd-temp-toggle' ).text( '🌡️ ' + tempLabel() );
            $( '#gd-height-toggle' ).text( '📏 ' + heightLabel() );
            return;
        }
        const $tBtn = $( `<button type="button" id="gd-temp-toggle"   class="gd-btn gd-btn-sm gd-unit-toggle">🌡️ ${tempLabel()}</button>` );
        const $hBtn = $( `<button type="button" id="gd-height-toggle" class="gd-btn gd-btn-sm gd-unit-toggle">📏 ${heightLabel()}</button>` );
        $( '#gd-modal-weeks .gd-modal-header' ).append( $tBtn ).append( $hBtn );
    }

    // Temp toggle
    $( document ).on( 'click', '#gd-temp-toggle', function() {
        const $inp = $( '[name=temp_c]' );
        const cur  = $inp.val();
        if ( tempUnit === 'F' ) { tempUnit = 'C'; if ( cur ) $inp.val( fToC(cur) ); }
        else                    { tempUnit = 'F'; if ( cur ) $inp.val( cToF(cur) ); }
        localStorage.setItem( 'gd_temp_unit', tempUnit );
        $( this ).text( '🌡️ ' + tempLabel() );
        $( '#gd-temp-unit-label' ).text( 'Temp (' + tempLabel() + ')' );
    } );

    // Height toggle
    $( document ).on( 'click', '#gd-height-toggle', function() {
        const $inp = $( '#gd-height-input' );
        const cur  = $inp.val();
        if ( heightUnit === 'in' ) { heightUnit = 'cm'; if ( cur ) $inp.val( inToCm(cur) ); }
        else                       { heightUnit = 'in'; if ( cur ) $inp.val( cmToIn(cur) ); }
        localStorage.setItem( 'gd_height_unit', heightUnit );
        $( this ).text( '📏 ' + heightLabel() );
        $( '#gd-height-unit-label' ).text( 'Plant Height (' + heightLabel() + ')' );
        $( '#gd-height-input' ).attr( 'placeholder', heightPlaceholder() );
    } );

    /* ===================================================================
       UTILITIES
    =================================================================== */
    function post( action, data ) {
        return $.post( AJAX, { ...data, action, nonce: NONCE } );
    }
    function showMsg( $el, msg, isError ) {
        $el.text( msg ).css( 'color', isError ? '#e84040' : '#4caf50' ).show();
    }
    function openModal( id )  { $( '#' + id ).fadeIn( 200 );  }
    function closeModal( id ) { $( '#' + id ).fadeOut( 200 ); }

    $( document ).on( 'click', '.gd-modal-close, [data-close]', function() {
        const target = $( this ).data( 'close' ) || $( this ).closest( '.gd-modal' ).attr( 'id' );
        closeModal( target );
    } );
    $( document ).on( 'click', '.gd-modal-overlay', function() {
        $( this ).closest( '.gd-modal' ).fadeOut( 200 );
    } );

    function stageIcon( stage ) {
        return { germination:'🌱', seedling:'🌿', vegetative:'🍃', flowering:'🌸', flushing:'💧', harvest:'🌾' }[ stage ] || '🌿';
    }
    function statusBadge( s ) {
        return { active:'🟢 Active', harvested:'🌾 Harvested', failed:'💀 Failed' }[ s ] || s;
    }
    function formatDate( str ) {
        if ( ! str ) return '';
        return new Date( str ).toLocaleDateString( 'en-US', { year:'numeric', month:'short', day:'numeric' } );
    }
    function renderStars( r ) {
        return r ? '★'.repeat(r) + '☆'.repeat(5-r) : '';
    }

    /* ===================================================================
       FIX 1: DIARY STORE – objects keyed by ID, no JSON in HTML attrs
    =================================================================== */
    const diaryStore = {};

    /* ===================================================================
       DASHBOARD: Diary List
    =================================================================== */
    if ( $( '#gd-diary-grid' ).length ) loadDiaries();

    function loadDiaries() {
        if ( ! IS_LOGGED ) {
            $( '#gd-diary-grid' ).html( '<div class="gd-notice">Please log in to view your diaries.</div>' );
            return;
        }
        post( 'growdiary_get_diaries' ).done( function( res ) {
            if ( res.success ) {
                res.data.forEach( d => { diaryStore[ d.id ] = d; } );
                renderDiaryGrid( res.data, '#gd-diary-grid', true );
            } else {
                $( '#gd-diary-grid' ).html( '<div class="gd-notice gd-notice--error">Could not load diaries.</div>' );
            }
        } );
    }

    function renderDiaryGrid( diaries, selector, isOwner ) {
        const $grid = $( selector );
        if ( ! diaries.length ) {
            $grid.html( '<div class="gd-empty">No grows yet. Start your first diary! 🌱</div>' );
            return;
        }
        let html = '';
        diaries.forEach( function( d ) {
            const img = d.cover_image
                ? `<div class="gd-card-img" style="background-image:url('${d.cover_image}')"></div>`
                : `<div class="gd-card-img gd-card-img--placeholder">🌿</div>`;

            html += `
            <div class="gd-card" data-id="${d.id}">
              ${img}
              <div class="gd-card-body">
                <div class="gd-card-tags">
                  <span class="gd-tag gd-tag--status">${statusBadge(d.status)}</span>
                  <span class="gd-tag">${d.week_count}/20 weeks</span>
                  ${d.visibility === 'public'
                    ? '<span class="gd-tag gd-tag--public">🌍 Public</span>'
                    : '<span class="gd-tag gd-tag--private">🔒 Private</span>'}
                </div>
                <h3 class="gd-card-title">${escHtml(d.title)}</h3>
                ${d.strain ? `<p class="gd-card-strain">🧬 ${escHtml(d.strain)}</p>` : ''}
                <div class="gd-card-meta">
                  ${d.light_type  ? `<span>💡 ${escHtml(d.light_type)}${d.light_watts ? ' ' + d.light_watts + 'W' : ''}</span>` : ''}
                  ${d.grow_medium ? `<span>🪴 ${escHtml(d.grow_medium)}</span>` : ''}
                  ${d.start_date  ? `<span>📅 ${formatDate(d.start_date)}</span>` : ''}
                </div>
                <div class="gd-card-actions">
                  <button class="gd-btn gd-btn-sm gd-btn-primary gd-open-weeks"
                    data-id="${d.id}" data-title="${escHtml(d.title)}">📋 Weekly Log</button>
                  ${isOwner ? `
                    <button class="gd-btn gd-btn-sm gd-edit-diary" data-diary-id="${d.id}">✏️ Edit</button>
                    <button class="gd-btn gd-btn-sm gd-btn-danger gd-delete-diary"
                      data-id="${d.id}" data-title="${escHtml(d.title)}">🗑️</button>
                  ` : `
                    <a href="${GrowDiary.view_url}?diary_id=${d.id}" class="gd-btn gd-btn-sm">👁️ View</a>
                  `}
                </div>
              </div>
            </div>`;
        } );
        $grid.html( html );
    }

    /* ===================================================================
       CREATE / EDIT DIARY
       editingDiaryId holds the ID of the diary being edited, or null
       when creating. Never stored in a form field — no reset() races.
    =================================================================== */
    let editingDiaryId = null;

    $( '#gd-open-create' ).on( 'click', function() {
        editingDiaryId = null;
        resetDiaryForm();
        $( '#gd-modal-diary-title' ).text( 'Start a New Grow' );
        $( '#gd-status-field' ).hide();
        openModal( 'gd-modal-diary' );
    } );

    function resetDiaryForm() {
        $( '#gd-form-diary' )[0].reset();
        $( '#gd-cover-preview' ).hide();
        $( '#gd-diary-msg' ).hide();
    }

    /* --- Edit: reads full object from diaryStore by ID only --- */
    $( document ).on( 'click', '.gd-edit-diary', function() {
        const id = parseInt( $( this ).data( 'diary-id' ), 10 );
        const d  = diaryStore[ id ];
        if ( ! d ) {
            alert( 'Diary data not found – please refresh and try again.' );
            return;
        }

        editingDiaryId = id;   // ← set BEFORE reset so it is never cleared
        resetDiaryForm();

        $( '#gd-modal-diary-title' ).text( 'Edit Grow: ' + d.title );
        $( '#gd-status-field' ).show();

        const $f = $( '#gd-form-diary' );
        $f.find( '[name=title]' ).val( d.title );
        $f.find( '[name=strain]' ).val( d.strain );
        $f.find( '[name=grow_medium]' ).val( d.grow_medium );
        $f.find( '[name=light_type]' ).val( d.light_type );
        $f.find( '[name=light_watts]' ).val( d.light_watts );
        $f.find( '[name=grow_space]' ).val( d.grow_space );
        $f.find( '[name=nutrients]' ).val( d.nutrients );
        $f.find( '[name=start_date]' ).val( d.start_date ? d.start_date.substring( 0, 10 ) : '' );
        $f.find( '[name=cover_image]' ).val( d.cover_image );
        $f.find( '[name=visibility]' ).val( d.visibility );
        $f.find( '[name=status]' ).val( d.status );
        $f.find( '[name=notes]' ).val( d.notes );

        if ( d.cover_image ) {
            $( '#gd-cover-img' ).attr( 'src', d.cover_image );
            $( '#gd-cover-preview' ).show();
        }

        openModal( 'gd-modal-diary' );
    } );

    /* --- Save diary (create or update) --- */
    $( '#gd-form-diary' ).on( 'submit', function( e ) {
        e.preventDefault();
        const $msg   = $( '#gd-diary-msg' );
        const action = editingDiaryId ? 'growdiary_update_diary' : 'growdiary_create_diary';

        const formData       = formToObj( $( this ) );
        formData.diary_id    = editingDiaryId || '';   // null → '' → create branch on server

        $msg.text( 'Saving…' ).css( 'color', '#888' ).show();

        post( action, formData ).done( function( res ) {
            if ( res.success ) {
                showMsg( $msg, res.data.message, false );
                setTimeout( function() {
                    closeModal( 'gd-modal-diary' );
                    editingDiaryId = null;
                    loadDiaries();
                }, 700 );
            } else {
                showMsg( $msg, res.data.message, true );
            }
        } );
    } );

    /* ===================================================================
       DELETE DIARY
    =================================================================== */
    let pendingDeleteDiaryId = null;

    $( document ).on( 'click', '.gd-delete-diary', function() {
        pendingDeleteDiaryId = $( this ).data( 'id' );
        $( '#gd-confirm-msg' ).text( `Delete "${$( this ).data('title')}" and all its weekly entries? This cannot be undone.` );
        $( '#gd-confirm-yes' ).data( 'type', 'diary' );
        openModal( 'gd-modal-confirm' );
    } );

    $( '#gd-confirm-yes' ).on( 'click', function() {
        const type = $( this ).data( 'type' );
        if ( type === 'diary' && pendingDeleteDiaryId ) {
            post( 'growdiary_delete_diary', { diary_id: pendingDeleteDiaryId } ).done( function( res ) {
                closeModal( 'gd-modal-confirm' );
                if ( res.success ) { delete diaryStore[ pendingDeleteDiaryId ]; loadDiaries(); }
            } );
        } else if ( type === 'week' ) {
            doDeleteWeek();
        }
    } );

    /* ===================================================================
       WEEKLY LOG MODAL
    =================================================================== */
    let currentDiaryId   = null;
    let currentWeek      = 1;
    let currentWeeksData = {};

    $( document ).on( 'click', '.gd-open-weeks', function() {
        currentDiaryId   = $( this ).data( 'id' );
        currentWeeksData = {};
        $( '#gd-weeks-title' ).text( 'Weekly Log – ' + $( this ).data( 'title' ) );
        $( '#gd-week-diary-id' ).val( currentDiaryId );

        loadWeeks( currentDiaryId, function() {
            injectWeekToggles();
            // Set label text to match current unit prefs
            $( '#gd-temp-unit-label' ).text( 'Temp (' + tempLabel() + ')' );
            $( '#gd-height-unit-label' ).text( 'Plant Height (' + heightLabel() + ')' );
            $( '#gd-height-input' ).attr( 'placeholder', heightPlaceholder() );
            buildWeekTabs();
            selectWeek( 1 );
            openModal( 'gd-modal-weeks' );
        } );
    } );

    function loadWeeks( diaryId, cb ) {
        post( 'growdiary_get_diary', { diary_id: diaryId } ).done( function( res ) {
            if ( res.success ) {
                res.data.weeks.forEach( w => { currentWeeksData[ w.week_number ] = w; } );
            }
            if ( cb ) cb();
        } );
    }

    function buildWeekTabs() {
        let html = '';
        for ( let i = 1; i <= 20; i++ ) {
            const filled = !! currentWeeksData[i];
            html += `<button class="gd-week-tab ${filled ? 'gd-week-tab--filled' : ''}" data-week="${i}">W${i}</button>`;
        }
        $( '#gd-week-tabs' ).html( html );
    }

    $( document ).on( 'click', '.gd-week-tab', function() {
        $( '.gd-week-tab' ).removeClass( 'gd-week-tab--active' );
        $( this ).addClass( 'gd-week-tab--active' );
        selectWeek( parseInt( $( this ).data( 'week' ), 10 ) );
    } );

    function selectWeek( weekNum ) {
        currentWeek = weekNum;
        $( '#gd-week-number' ).val( weekNum );
        $( '#gd-week-msg' ).hide();

        const d  = currentWeeksData[ weekNum ] || {};
        const $f = $( '#gd-form-week' );

        $f.find( '[name=stage]' ).val( d.stage || 'vegetative' );

        // FIX 2: height – display in user's chosen unit
        $f.find( '#gd-height-input' ).val( heightToDisplay( d.height_cm ) );

        $f.find( '[name=ph_water]' ).val( d.ph_water || '' );
        $f.find( '[name=ph_runoff]' ).val( d.ph_runoff || '' );
        $f.find( '[name=ec_ppm]' ).val( d.ec_ppm || '' );
        $f.find( '[name=temp_c]' ).val( tempToDisplay( d.temp_c ) );
        $f.find( '[name=humidity_pct]' ).val( d.humidity_pct || '' );
        
        $f.find( '[name=light_hours]' ).val( d.light_hours || '' );
        $f.find( '[name=watering_ml]' ).val( d.watering_ml || '' );
        $f.find( '[name=nutrients]' ).val( d.nutrients || '' );
        $f.find( '[name=observations]' ).val( d.observations || '' );
        $f.find( '[name=problems]' ).val( d.problems || '' );

        const rating = d.rating || 0;
        $( '#gd-rating-input' ).val( rating );
        updateStarUI( rating );

        const images = d.images || [];
        $( '#gd-week-images-json' ).val( JSON.stringify( images ) );
        renderWeekGallery( images );

        // Restore chart data if this week has sensor data, show preview
        $( '#gd-csv-preview-chart' ).remove();
        $( '#gd-csv-status' ).text('');
        if ( d.chart_data ) {
            $( '#gd-chart-data-json' ).val( JSON.stringify( d.chart_data ) );
            renderImportPreviewChart( d.chart_data.fields, d.chart_data.rows );
            $( '#gd-csv-status' ).html( '📡 Sensor data loaded' ).css( 'color', '#4caf50' );
        } else {
            $( '#gd-chart-data-json' ).val('');
        }

        $( '.gd-week-tab' ).removeClass( 'gd-week-tab--active' );
        $( `.gd-week-tab[data-week="${weekNum}"]` ).addClass( 'gd-week-tab--active' );
    }

    /* ===================================================================
       SAVE WEEK – convert display units back to cm / °C before POST
       Same diary_id safety fix applied here too.
    =================================================================== */
    $( '#gd-form-week' ).on( 'submit', function( e ) {
        e.preventDefault();
        const $msg     = $( '#gd-week-msg' );
        const formData = formToObj( $( this ) );

        // Restore hidden field values that formToObj skips
        formData.diary_id    = $( '#gd-week-diary-id' ).val();
        formData.week_number = $( '#gd-week-number' ).val();

        // Convert displayed units back to metric for storage
        if ( formData.temp_c    !== '' ) formData.temp_c    = displayToC( formData.temp_c );
        if ( formData.height_cm !== '' ) formData.height_cm = displayToCm( formData.height_cm );

        formData.images     = $( '#gd-week-images-json' ).val();
        formData.chart_data = $( '#gd-chart-data-json' ).val();

        $msg.text( 'Saving…' ).css( 'color', '#888' ).show();

        post( 'growdiary_save_week', formData ).done( function( res ) {
            if ( res.success ) {
                showMsg( $msg, res.data.message, false );
                loadWeeks( currentDiaryId, function() {
                    buildWeekTabs();
                    $( `.gd-week-tab[data-week="${currentWeek}"]` ).addClass( 'gd-week-tab--active' );
                } );
            } else {
                showMsg( $msg, res.data.message, true );
            }
        } );
    } );

    /* ===================================================================
       DELETE WEEK
    =================================================================== */
    $( '#gd-delete-week-btn' ).on( 'click', function() {
        if ( ! currentWeeksData[ currentWeek ] ) return;
        $( '#gd-confirm-msg' ).text( `Delete Week ${currentWeek} entry? This cannot be undone.` );
        $( '#gd-confirm-yes' ).data( 'type', 'week' );
        openModal( 'gd-modal-confirm' );
    } );

    function doDeleteWeek() {
        post( 'growdiary_delete_week', {
            diary_id:    currentDiaryId,
            week_number: currentWeek,
        } ).done( function( res ) {
            closeModal( 'gd-modal-confirm' );
            if ( res.success ) {
                delete currentWeeksData[ currentWeek ];
                buildWeekTabs();
                selectWeek( currentWeek );
            }
        } );
    }

    /* ===================================================================
       STAR RATING
    =================================================================== */
    $( document ).on( 'mouseenter', '#gd-star-rating span', function() { updateStarUI( $( this ).data('val') ); } );
    $( '#gd-star-rating' ).on( 'mouseleave', function() { updateStarUI( parseInt( $( '#gd-rating-input' ).val() ) || 0 ); } );
    $( document ).on( 'click', '#gd-star-rating span', function() {
        const val = $( this ).data('val');
        $( '#gd-rating-input' ).val( val );
        updateStarUI( val );
    } );
    function updateStarUI( r ) {
        $( '#gd-star-rating span' ).each( function() {
            $( this ).toggleClass( 'active', $( this ).data('val') <= r );
        } );
    }

    /* ===================================================================
       IMAGE UPLOAD (WP Media Uploader)
    =================================================================== */
    $( '#gd-cover-upload-btn' ).on( 'click', function(e) {
        e.preventDefault();
        openMediaUploader( function( url ) {
            $( '#gd-cover-image-url' ).val( url );
            $( '#gd-cover-img' ).attr( 'src', url );
            $( '#gd-cover-preview' ).show();
        } );
    } );
    $( '#gd-cover-image-url' ).on( 'change input', function() {
        const url = $( this ).val();
        if ( url ) { $( '#gd-cover-img' ).attr( 'src', url ); $( '#gd-cover-preview' ).show(); }
        else        { $( '#gd-cover-preview' ).hide(); }
    } );

    $( '#gd-week-upload-btn' ).on( 'click', function(e) {
        e.preventDefault();
        openMediaUploader( function( url ) {
            const images = JSON.parse( $( '#gd-week-images-json' ).val() || '[]' );
            images.push( url );
            $( '#gd-week-images-json' ).val( JSON.stringify( images ) );
            renderWeekGallery( images );
        } );
    } );

    function renderWeekGallery( images ) {
        const $g = $( '#gd-week-gallery' );
        if ( ! images.length ) { $g.html( '<p class="gd-no-photos">No photos yet</p>' ); return; }
        let html = '';
        images.forEach( function( url, i ) {
            html += `<div class="gd-gallery-thumb">
              <img src="${url}" alt="Week photo" class="gd-lb-trigger" data-src="${url}" data-index="${i}" data-set="gallery">
              <button type="button" class="gd-remove-photo" data-index="${i}">&times;</button>
            </div>`;
        } );
        $g.html( html );
    }

    $( document ).on( 'click', '.gd-remove-photo', function() {
        const idx    = $( this ).data( 'index' );
        const images = JSON.parse( $( '#gd-week-images-json' ).val() || '[]' );
        images.splice( idx, 1 );
        $( '#gd-week-images-json' ).val( JSON.stringify( images ) );
        renderWeekGallery( images );
    } );

    function openMediaUploader( cb ) {
        const frame = wp.media( { title: 'Select Image', button: { text: 'Use Image' }, multiple: false } );
        frame.on( 'select', function() { cb( frame.state().get('selection').first().toJSON().url ); } );
        frame.open();
    }

    /* ===================================================================
       MARS HYDRO CSV IMPORT
       Parses the exported THP CSV, computes week averages to auto-fill
       the form, and stores a downsampled dataset for chart rendering.
    =================================================================== */
    $( '#gd-csv-upload-btn' ).on( 'click', function() {
        $( '#gd-csv-file-input' ).val('').trigger('click');
    } );

    $( '#gd-csv-file-input' ).on( 'change', function() {
        const file = this.files[0];
        if ( ! file ) return;
        const $status = $( '#gd-csv-status' );
        $status.text( 'Parsing…' ).css( 'color', '#888' );

        const reader = new FileReader();
        reader.onload = function( e ) {
            try {
                const result = parseMarsHydroCSV( e.target.result );
                if ( ! result || ! result.rows.length ) {
                    $status.text( '⚠️ No valid data rows found in CSV.' ).css( 'color', '#e84040' );
                    return;
                }

                // Auto-fill form averages
                const avg = key => {
                    const vals = result.rows.map( r => r[key] ).filter( v => v !== null && ! isNaN(v) );
                    return vals.length ? vals.reduce((a,b)=>a+b,0) / vals.length : null;
                };

                const avgTempC   = avg('temp_c');
                const avgHumidity = avg('humidity');
                const avgVpdVal  = avg('vpd');

                const $f = $( '#gd-form-week' );

                if ( avgTempC !== null ) {
                    // Convert to display unit
                    const displayTemp = tempUnit === 'F' ? cToF( avgTempC ) : avgTempC.toFixed(1);
                    $f.find( '[name=temp_c]' ).val( displayTemp );
                }
                if ( avgHumidity !== null ) {
                    $f.find( '[name=humidity_pct]' ).val( Math.round( avgHumidity ) );
                }
                if ( avgVpdVal !== null ) {
                    $f.find( '[name=vpd]' ).val( avgVpdVal.toFixed(2) );
                }

                // Downsample to ~120 points for chart storage
                const chartRows = downsample( result.rows, 120 );

                // Store as JSON for save
                $( '#gd-chart-data-json' ).val( JSON.stringify( {
                    fields: result.fields,
                    rows:   chartRows,
                } ) );

                // Preview mini-chart inside the modal
                renderImportPreviewChart( result.fields, chartRows );

                const d = result.rows.length;
                $status.html(
                    `✅ Imported <strong>${d}</strong> readings over <strong>${result.duration}</strong> &mdash; ` +
                    `Avg Temp: <strong>${avgTempC !== null ? (tempUnit==='F'?cToF(avgTempC):avgTempC.toFixed(1))+(tempUnit==='F'?'°F':'°C') : '–'}</strong> &nbsp;` +
                    `RH: <strong>${avgHumidity !== null ? Math.round(avgHumidity)+'%' : '–'}</strong> &nbsp;` +
                    `VPD: <strong>${avgVpdVal !== null ? avgVpdVal.toFixed(2)+' kPa' : '–'}</strong>`
                ).css( 'color', '#4caf50' );

            } catch ( err ) {
                console.error( 'CSV parse error:', err );
                $status.text( '⚠️ Could not parse CSV: ' + err.message ).css( 'color', '#e84040' );
            }
        };
        reader.readAsText( file );
    } );

    function parseMarsHydroCSV( text ) {
        // Strip BOM
        text = text.replace( /^\uFEFF/, '' );
        const lines = text.split( /\r?\n/ ).filter( l => l.trim() );
        if ( lines.length < 2 ) return null;

        const headers = lines[0].split(',').map( h => h.trim() );

        // Map header names to our field keys
        const colMap = {};
        headers.forEach( (h, i) => {
            const hl = h.toLowerCase();
            if ( hl.includes('temperature') && hl.includes('°c') ) colMap.temp_c   = i;
            if ( hl.includes('temperature') && hl.includes('°f') ) colMap.temp_f   = i;
            if ( hl === 'humidity' )                                 colMap.humidity = i;
            if ( hl === 'vpd' )                                      colMap.vpd      = i;
            if ( hl === 'timestamp' )                                colMap.ts       = i;
            // Future fields: ppfd, ec, co2, lux etc
            if ( hl === 'ppfd' )                                     colMap.ppfd     = i;
            if ( hl === 'ec' )                                       colMap.ec       = i;
            if ( hl === 'co2' )                                      colMap.co2      = i;
        } );

        const fields = Object.keys( colMap ).filter( k => k !== 'ts' );
        const rows   = [];
        let   firstTs = null, lastTs = null;

        for ( let i = 1; i < lines.length; i++ ) {
            const cols = lines[i].split(',');
            // Skip rows with no sensor data (all blank except device/ts)
            const hasData = fields.some( f => cols[ colMap[f] ] && cols[ colMap[f] ].trim() !== '' );
            if ( ! hasData ) continue;

            const ts = colMap.ts !== undefined ? cols[ colMap.ts ]?.trim() : null;
            const row = { ts };
            fields.forEach( f => {
                const v = cols[ colMap[f] ]?.trim();
                row[f] = ( v !== '' && v !== undefined ) ? parseFloat(v) : null;
            } );

            rows.push( row );
            if ( ! firstTs ) firstTs = ts;
            lastTs = ts;
        }

        // Human-readable duration
        let duration = '';
        if ( firstTs && lastTs ) {
            const ms = new Date(lastTs) - new Date(firstTs);
            const hrs = Math.round( ms / 3600000 );
            duration = hrs >= 24 ? `${Math.round(hrs/24)}d ${hrs%24}h` : `${hrs}h`;
        }

        return { fields, rows, duration };
    }

    function downsample( rows, target ) {
        if ( rows.length <= target ) return rows;
        const step   = rows.length / target;
        const result = [];
        for ( let i = 0; i < target; i++ ) {
            result.push( rows[ Math.min( Math.round( i * step ), rows.length - 1 ) ] );
        }
        return result;
    }

    /* Mini preview chart inside the weekly log modal */
    let importChartInstance = null;

    function renderImportPreviewChart( fields, rows ) {
        $( '#gd-csv-preview-chart' ).remove();
        const $wrap = $( '<div id="gd-csv-preview-chart" style="margin-top:12px;"></div>' );
        $( '#gd-csv-upload-btn' ).closest( '.gd-field' ).append( $wrap );
        if ( importChartInstance ) {
            importChartInstance.forEach( c => c && c.destroy() );
            importChartInstance = null;
        }
        importChartInstance = buildThreeCharts( $wrap, fields, rows, 'gd-csv-canvas', 180 );
    }

    /* ===================================================================
       FIX 3: LIGHTBOX
       Two contexts:
         a) Week gallery thumbnails inside the edit modal
         b) Timeline photos on the view-diary page
       Both use the same #gd-lightbox overlay (present in both templates).
       Images within a "set" (same week / same timeline card) are
       navigable with prev/next arrows.
    =================================================================== */
    let lbImages = [];
    let lbIndex  = 0;

    function lbOpen( imgs, idx ) {
        lbImages = imgs;
        lbIndex  = idx;
        lbShow();
        $( '#gd-lightbox' ).fadeIn( 180 );
        $( 'body' ).addClass( 'gd-lb-open' );
    }
    function lbClose() {
        $( '#gd-lightbox' ).fadeOut( 180 );
        $( 'body' ).removeClass( 'gd-lb-open' );
    }
    function lbShow() {
        $( '#gd-lightbox-img' ).attr( 'src', lbImages[ lbIndex ] );
        $( '#gd-lightbox-counter' ).text( (lbIndex + 1) + ' / ' + lbImages.length );
        $( '#gd-lightbox-prev, #gd-lightbox-next' ).toggle( lbImages.length > 1 );
    }
    function lbPrev() { lbIndex = ( lbIndex - 1 + lbImages.length ) % lbImages.length; lbShow(); }
    function lbNext() { lbIndex = ( lbIndex + 1 ) % lbImages.length; lbShow(); }

    $( document ).on( 'click', '#gd-lightbox-close, #gd-lightbox-overlay', lbClose );
    $( document ).on( 'click', '#gd-lightbox-prev', function(e) { e.stopPropagation(); lbPrev(); } );
    $( document ).on( 'click', '#gd-lightbox-next', function(e) { e.stopPropagation(); lbNext(); } );
    $( document ).on( 'keydown', function(e) {
        if ( ! $( '#gd-lightbox' ).is(':visible') ) return;
        if ( e.key === 'ArrowLeft'  ) lbPrev();
        if ( e.key === 'ArrowRight' ) lbNext();
        if ( e.key === 'Escape'     ) lbClose();
    } );

    // Week pip click → scroll to that week's timeline card (view-diary page)
    $( document ).on( 'click', '.gd-week-pip--filled', function() {
        const wn  = $( this ).data( 'week' );
        const el  = document.getElementById( 'gd-week-card-' + wn );
        if ( el ) {
            el.scrollIntoView( { behavior: 'smooth', block: 'start' } );
        }
    } );

    // Trigger from week edit gallery thumbs
    $( document ).on( 'click', '.gd-lb-trigger', function(e) {
        e.preventDefault();
        e.stopPropagation(); // don't also fire remove-photo etc.
        const $set = $( this ).closest( '.gd-image-gallery, .gd-timeline-photos' );
        const imgs = $set.find( '.gd-lb-trigger' ).map( function() {
            return $( this ).data( 'src' ) || $( this ).attr( 'src' );
        } ).get();
        const idx  = parseInt( $( this ).data( 'index' ) || 0, 10 );
        lbOpen( imgs, idx );
    } );

    // Note: .gd-timeline-img also carries .gd-lb-trigger so the handler above already handles it.

    /* ===================================================================
       VIEW DIARY PAGE
    =================================================================== */
    if ( $( '#gd-view-app' ).length ) {
        const params  = new URLSearchParams( window.location.search );
        const diaryId = params.get( 'diary_id' );
        if ( ! diaryId ) {
            $( '#gd-view-loading' ).hide();
            $( '#gd-view-error' ).text( 'No diary specified.' ).show();
        } else {
            post( 'growdiary_get_diary', { diary_id: diaryId } ).done( function( res ) {
                $( '#gd-view-loading' ).hide();
                console.log( 'GrowDiary view response:', JSON.stringify( res ) );
                if ( res.success ) renderDiaryView( res.data.diary, res.data.weeks );
                else $( '#gd-view-error' ).text( res.data.message ).show();
            } ).fail( function( xhr, status, err ) {
                $( '#gd-view-loading' ).hide();
                console.error( 'GrowDiary AJAX fail:', status, err, xhr.responseText );
                $( '#gd-view-error' ).text( 'Could not load diary. Check console for details.' ).show();
            } );
        }
    }

    function renderDiaryView( d, weeks ) {
        /* ---------- Hero ---------- */
        $( '#gd-view-title' ).text( d.title );

        const daysRunning = d.start_date
            ? Math.floor( ( Date.now() - new Date(d.start_date) ) / 86400000 )
            : null;
        let metaParts = [ `By ${d.author_name}` ];
        if ( d.start_date ) metaParts.push( `Started ${formatDate(d.start_date)}` );
        if ( daysRunning !== null && d.status === 'active' ) metaParts.push( `Day ${daysRunning}` );
        $( '#gd-view-meta' ).text( metaParts.join('  ·  ') );

        const tags = [
            d.strain      ? `🧬 ${d.strain}` : '',
            d.grow_medium ? `🪴 ${d.grow_medium}` : '',
            d.light_type  ? `💡 ${d.light_type}${d.light_watts ? ' ' + d.light_watts + 'W' : ''}` : '',
            d.grow_space  ? `📐 ${d.grow_space}` : '',
            d.status      ? `${{ active:'🟢', harvested:'🌾', failed:'💀' }[d.status] || ''} ${d.status}` : '',
        ].filter(Boolean).map( t => `<span class="gd-tag">${t}</span>` ).join('');
        $( '#gd-view-tags' ).html( tags );
        if ( d.cover_image ) $( '#gd-diary-hero' ).css( 'background-image', `url(${d.cover_image})` );

        $( '#gd-view-content' ).show();

        /* ---------- Setup recap cards ---------- */
        const recapItems = [
            d.strain       ? { icon:'🧬', label:'Strain',      val: d.strain }       : null,
            d.grow_medium  ? { icon:'🪴', label:'Medium',      val: d.grow_medium }  : null,
            d.light_type   ? { icon:'💡', label:'Light',       val: d.light_type + (d.light_watts ? ' · ' + d.light_watts + 'W' : '') } : null,
            d.grow_space   ? { icon:'📐', label:'Space',       val: d.grow_space }   : null,
            d.nutrients    ? { icon:'🧪', label:'Nutrients',   val: d.nutrients }    : null,
            d.start_date   ? { icon:'📅', label:'Start Date',  val: formatDate(d.start_date) } : null,
        ].filter(Boolean);

        if ( recapItems.length ) {
            const recapHtml = recapItems.map( item => `
                <div class="gd-recap-card">
                  <div class="gd-recap-icon">${item.icon}</div>
                  <div class="gd-recap-body">
                    <small>${item.label}</small>
                    <strong>${escHtml(item.val)}</strong>
                  </div>
                </div>` ).join('');
            $( '#gd-recap-grid' ).html( recapHtml );
        }

        if ( d.notes ) {
            $( '#gd-recap-grid' ).append(
                `<div class="gd-recap-notes"><strong>📋 Grow Notes:</strong><p>${escHtml(d.notes)}</p></div>`
            );
        }

        /* ---------- No weeks ---------- */
        if ( ! weeks.length ) {
            $( '#gd-no-weeks' ).show();
            return;
        }

        /* ---------- Progress bar ---------- */
        const logged = weeks.length;
        const pct    = Math.round( logged / 20 * 100 );
        $( '#gd-progress-count' ).text( `${logged} / 20 weeks logged` );
        $( '#gd-progress-fill' ).css( 'width', pct + '%' );

        // Stage colour pips
        const stageColors = {
            germination:'#8bc34a', seedling:'#4caf50', vegetative:'#2e7d32',
            flowering:'#e91e63', flushing:'#2196f3', harvest:'#ff9800'
        };
        // Build a map of week→data for fast lookup
        const weekMap = {};
        weeks.forEach( w => { weekMap[ w.week_number ] = w; } );

        let pipHtml = '';
        for ( let i = 1; i <= 20; i++ ) {
            const w     = weekMap[i];
            const color = w ? ( stageColors[w.stage] || '#4caf50' ) : '#d0e4d0';
            const label = w ? `W${i}: ${w.stage}${w.rating ? ' · ' + '★'.repeat(w.rating) : ''}` : `W${i}: not logged`;
            pipHtml += `<div class="gd-week-pip ${w ? 'gd-week-pip--filled' : ''}"
              style="background:${color}" title="${label}" data-week="${i}"></div>`;
        }
        $( '#gd-week-pip-row' ).html( pipHtml );

        // Pip click handler is bound once at document level (see bottom of file)

        /* ---------- Grow summary stats ---------- */
        const withHeight = weeks.filter( w => w.height_cm );
        const withTemp   = weeks.filter( w => w.temp_c );
        const withHum    = weeks.filter( w => w.humidity_pct );
        const withPh     = weeks.filter( w => w.ph_water );
        const withVpd    = weeks.filter( w => w.vpd );

        const ratings    = weeks.filter( w => w.rating ).map( w => w.rating );
        const allPhotos  = weeks.reduce( (acc, w) => acc + (w.images||[]).length, 0 );

        const avg = arr => arr.length ? ( arr.reduce((a,b)=>a+b,0) / arr.length ) : null;
        const max = arr => arr.length ? Math.max(...arr) : null;

        const maxHeightCm  = max( withHeight.map(w=>parseFloat(w.height_cm)) );
        const avgTempC     = avg( withTemp.map(w=>parseFloat(w.temp_c)) );
        const avgHum       = avg( withHum.map(w=>parseFloat(w.humidity_pct)) );
        const avgPh        = avg( withPh.map(w=>parseFloat(w.ph_water)) );
        const avgVpd       = avg( withVpd.map(w=>parseFloat(w.vpd)) );

        const avgRating    = avg( ratings );

        const summaryItems = [];
        if ( maxHeightCm !== null ) {
            const dispH = heightUnit === 'in' ? cmToIn(maxHeightCm) + ' in' : maxHeightCm.toFixed(1) + ' cm';
            summaryItems.push({ icon:'📏', label:'Max Height', val: dispH });
        }
        if ( avgTempC !== null ) {
            const dispT = tempUnit === 'F' ? cToF(avgTempC) + '°F' : avgTempC.toFixed(1) + '°C';
            summaryItems.push({ icon:'🌡️', label:'Avg Temp', val: dispT });
        }
        if ( avgHum !== null ) summaryItems.push({ icon:'💧', label:'Avg Humidity', val: avgHum.toFixed(0) + '%' });
        if ( avgVpd !== null ) summaryItems.push({ icon:'🌬️', label:'Avg VPD', val: avgVpd.toFixed(2) + ' kPa' });
        if ( avgPh  !== null ) summaryItems.push({ icon:'⚗️', label:'Avg pH', val: avgPh.toFixed(2) });
		
        if ( avgRating !== null ) summaryItems.push({ icon:'⭐', label:'Avg Rating', val: avgRating.toFixed(1) + ' / 5' });
        summaryItems.push({ icon:'📸', label:'Total Photos', val: allPhotos });
        summaryItems.push({ icon:'📋', label:'Weeks Logged', val: `${logged} / 20` });

        // Stage breakdown
        const stageCounts = {};
        weeks.forEach( w => { stageCounts[w.stage] = (stageCounts[w.stage]||0) + 1; } );
        const stageStr = Object.entries(stageCounts).map(([s,c])=>`${stageIcon(s)} ${s} ×${c}`).join('  ');
        summaryItems.push({ icon:'🔬', label:'Stage Breakdown', val: stageStr, wide: true });

        const sumHtml = summaryItems.map( item =>
            `<div class="gd-summary-card ${item.wide ? 'gd-summary-card--wide' : ''}">
              <div class="gd-summary-icon">${item.icon}</div>
              <div><small>${item.label}</small><strong>${item.val}</strong></div>
            </div>` ).join('');
        $( '#gd-summary-grid' ).html( sumHtml );
        $( '#gd-summary-stats' ).show();

        /* ---------- Unit toggles ---------- */
        $( '#gd-view-temp-toggle'  ).text( `🌡️ Switch to ${tempUnit === 'F' ? '°C' : '°F'}` );
        $( '#gd-view-height-toggle').text( `📏 Switch to ${heightUnit === 'in' ? 'cm' : 'in'}` );
        $( '#gd-view-toggles' ).show();

        $( '#gd-view-temp-toggle' ).off('click').on( 'click', function() {
            tempUnit = tempUnit === 'F' ? 'C' : 'F';
            localStorage.setItem( 'gd_temp_unit', tempUnit );
            $( this ).text( `🌡️ Switch to ${tempUnit === 'F' ? '°C' : '°F'}` );
            renderWeekTimeline( weeks, weekMap );
            // re-render summary too
            renderDiaryView.__refreshSummary && renderDiaryView.__refreshSummary();
        } );
        $( '#gd-view-height-toggle' ).off('click').on( 'click', function() {
            heightUnit = heightUnit === 'in' ? 'cm' : 'in';
            localStorage.setItem( 'gd_height_unit', heightUnit );
            $( this ).text( `📏 Switch to ${heightUnit === 'in' ? 'cm' : 'in'}` );
            renderWeekTimeline( weeks, weekMap );
        } );

        /* ---------- Timeline ---------- */
        $( '#gd-weeks-heading' ).show();
        renderWeekTimeline( weeks, weekMap );
    }

    function renderWeekTimeline( weeks, weekMap ) {
        let html = '';
        weeks.forEach( function( w ) {
            const imgs = ( w.images || [] ).map( ( url, i ) =>
                `<img src="${url}" class="gd-timeline-img gd-lb-trigger"
                  data-src="${url}" data-index="${i}" alt="Week ${w.week_number}">`
            ).join('');

            let tempDisplay = '';
            if ( w.temp_c !== null && w.temp_c !== '' ) {
                const val = tempUnit === 'F' ? cToF( w.temp_c ) : parseFloat(w.temp_c).toFixed(1);
                tempDisplay = `<div class="gd-stat"><span>🌡️</span><strong>${val}${tempLabel()}</strong><small>Temp</small></div>`;
            }
            let heightDisplay = '';
            if ( w.height_cm !== null && w.height_cm !== '' ) {
                const val = heightUnit === 'in' ? cmToIn( w.height_cm ) : parseFloat(w.height_cm).toFixed(1);
                heightDisplay = `<div class="gd-stat"><span>📏</span><strong>${val} ${heightLabel()}</strong><small>Height</small></div>`;
            }

            // Watering in a friendlier unit
            let waterDisplay = '';
            if ( w.watering_ml ) {
                waterDisplay = `<div class="gd-stat"><span>🚿</span><strong>${w.watering_ml} ml</strong><small>Watered</small></div>`;
            }

            html += `
            <div class="gd-timeline-item" id="gd-week-card-${w.week_number}">
              <div class="gd-timeline-badge">${stageIcon(w.stage)}</div>
              <div class="gd-timeline-card">
                <div class="gd-timeline-header">
                  <span class="gd-timeline-week">Week ${w.week_number}</span>
                  <span class="gd-tag gd-tag--stage">${w.stage}</span>
                  ${w.rating ? `<span class="gd-stars">${renderStars(w.rating)}</span>` : ''}
                  ${w.ph_runoff ? `<span class="gd-tag gd-tag--ph">pH out: ${w.ph_runoff}</span>` : ''}
                </div>

                <div class="gd-timeline-stats">
                  ${heightDisplay}
                  ${tempDisplay}
                  ${w.humidity_pct ? `<div class="gd-stat"><span>💧</span><strong>${w.humidity_pct}%</strong><small>RH</small></div>` : ''}
                  ${w.vpd         ? `<div class="gd-stat"><span>🌬️</span><strong>${parseFloat(w.vpd).toFixed(2)} kPa</strong><small>VPD</small></div>` : ''}
                  ${w.ph_water    ? `<div class="gd-stat"><span>⚗️</span><strong>${w.ph_water}</strong><small>pH in</small></div>` : ''}
                  ${w.ec_ppm      ? `<div class="gd-stat"><span>🔋</span><strong>${w.ec_ppm} ppm</strong><small>EC/PPM</small></div>` : ''}
                  ${w.light_hours ? `<div class="gd-stat"><span>☀️</span><strong>${w.light_hours}h</strong><small>Light</small></div>` : ''}
                  ${waterDisplay}
                </div>

                ${w.nutrients    ? `<div class="gd-timeline-text"><strong>🧪 Nutrients:</strong><p>${escHtml(w.nutrients)}</p></div>` : ''}
                ${w.observations ? `<div class="gd-timeline-text"><strong>📝 Observations:</strong><p>${escHtml(w.observations)}</p></div>` : ''}
                ${w.problems     ? `<div class="gd-timeline-text gd-timeline-text--warn"><strong>⚠️ Problems:</strong><p>${escHtml(w.problems)}</p></div>` : ''}

                ${w.chart_data ? `<div class="gd-sensor-chart-wrap"><div class="gd-sensor-chart-title">📡 Mars Hydro Sensor Data — Week ${w.week_number}</div><div class="gd-sensor-charts" id="gd-sensor-chart-wrap-${w.week_number}"></div></div>` : ''}

                ${imgs ? `<div class="gd-timeline-photos">${imgs}</div>` : ''}
              </div>
            </div>`;
        } );
        $( '#gd-timeline' ).html( html );

        // Render sensor charts for any weeks that have chart_data
        renderSensorCharts( weeks );
    }

    /* Render Chart.js charts for all weeks that have sensor data */
    const sensorChartInstances = {};

    /* ---------------------------------------------------------------
       buildThreeCharts – shared by both preview and view-diary.
       Chart 1: Temp (°C, 0-100 scale) + Humidity (%, 0-100 scale)
       Chart 2: VPD (kPa, 0-3 scale)
       Chart 3: Extra fields – PPFD, EC, CO₂ etc (auto-scaled)
    --------------------------------------------------------------- */
    function buildThreeCharts( $container, fields, rows, idPrefix, maxHeight ) {
        const labels = rows.map( r => r.ts ? r.ts.substring(5,16).replace('T',' ') : '' );
        const instances = [];

        const makeCanvas = ( id, title ) => {
            const $block = $( `<div class="gd-chart-block"><div class="gd-chart-subtitle">${title}</div><canvas id="${id}" style="max-height:${maxHeight}px"></canvas></div>` );
            $container.append( $block );
            return document.getElementById( id );
        };

        const makeLine = ( field, color, fill ) => ( {
            label:           { temp_c:'Temp (°C)', temp_f:'Temp (°F)', humidity:'Humidity (%)', vpd:'VPD (kPa)', ppfd:'PPFD (μmol)', ec:'EC', co2:'CO₂ (ppm)' }[field] || field,
            data:            rows.map( r => r[field] ),
            borderColor:     color,
            backgroundColor: fill || color + '18',
            borderWidth:     2,
            pointRadius:     0,
            tension:         0.35,
            fill:            !! fill,
        } );

        const baseOpts = ( yMin, yMax, tooltipRows ) => ( {
            responsive:  true,
            interaction: { mode: 'index', intersect: false },
            plugins: {
                legend:  { position: 'top', labels: { boxWidth: 12, font: { size: 11 } } },
                tooltip: { callbacks: { title: items => tooltipRows[ items[0].dataIndex ]?.ts || '' } },
            },
            scales: {
                x: { ticks: { maxTicksLimit: 10, font: { size: 10 }, maxRotation: 0 }, grid: { display: false } },
                y: {
                    min:      yMin,
                    max:      yMax,
                    ticks:    { font: { size: 10 } },
                    grid:     { color: 'rgba(0,0,0,.05)' },
                },
            },
        } );

        // --- Chart 1: Temp °C, Temp °F + Humidity (0–100 scale) ---
        const thFields = fields.filter( f => f === 'temp_c' || f === 'temp_f' || f === 'humidity' );
        if ( thFields.length ) {
            const cvs = makeCanvas( idPrefix + '-th', '🌡️ Temperature &amp; 💧 Humidity' );
            const ds  = [];
            if ( fields.includes('temp_c') )  ds.push( makeLine( 'temp_c',   '#e53935', '#e5393518' ) );
            if ( fields.includes('temp_f') )  ds.push( makeLine( 'temp_f',   '#ff7043', null ) );
            if ( fields.includes('humidity') ) ds.push( makeLine( 'humidity', '#1e88e5', '#1e88e518' ) );
            // If °F is present scale goes above 100, use auto; otherwise lock 0-100 for humidity readability
            const thMin = 0;
            const thMax = fields.includes('temp_f') ? undefined : 100;
            instances.push( new Chart( cvs.getContext('2d'), {
                type: 'line', data: { labels, datasets: ds },
                options: baseOpts( thMin, thMax, rows ),
            } ) );
        }

        // --- Chart 2: VPD (0–3 kPa) ---
        if ( fields.includes('vpd') ) {
            const cvs = makeCanvas( idPrefix + '-vpd', '🌬️ VPD (kPa)' );
            // Colour zones: green 0.8-1.5 ideal, red outside
            instances.push( new Chart( cvs.getContext('2d'), {
                type: 'line',
                data: { labels, datasets: [ makeLine( 'vpd', '#7b1fa2', '#7b1fa218' ) ] },
                options: {
                    ...baseOpts( 0, 3, rows ),
                    plugins: {
                        ...baseOpts( 0, 3, rows ).plugins,
                        annotation: undefined, // skip annotation plugin dependency
                    },
                },
            } ) );
        }

        // --- Chart 3: Extra fields (PPFD, EC, CO₂) auto-scaled ---
        const extraFields = fields.filter( f => !['temp_c','temp_f','humidity','vpd'].includes(f) );
        if ( extraFields.length ) {
            const palette = [ '#f57c00', '#00897b', '#6d4c41', '#546e7a' ];
            const cvs = makeCanvas( idPrefix + '-extra', '📊 ' + extraFields.map( f => ({ ppfd:'PPFD', ec:'EC', co2:'CO₂' }[f] || f) ).join(' &amp; ') );
            const ds  = extraFields.map( (f,i) => makeLine( f, palette[i % palette.length] ) );
            instances.push( new Chart( cvs.getContext('2d'), {
                type: 'line', data: { labels, datasets: ds },
                options: baseOpts( undefined, undefined, rows ),
            } ) );
        }

        return instances;
    }

    function renderSensorCharts( weeks ) {
        // Destroy old instances first
        Object.values( sensorChartInstances ).forEach( arr => arr && arr.forEach( c => c && c.destroy() ) );
        Object.keys( sensorChartInstances ).forEach( k => delete sensorChartInstances[k] );

        weeks.forEach( function( w ) {
            if ( ! w.chart_data || ! w.chart_data.rows ) return;
            const $wrap = $( '#gd-sensor-chart-wrap-' + w.week_number );
            if ( ! $wrap.length ) return;

            const { fields, rows } = w.chart_data;
            sensorChartInstances[ w.week_number ] = buildThreeCharts( $wrap, fields, rows, 'gd-sc-' + w.week_number, 220 );
        } );
    }

    /* ===================================================================
       COMMUNITY PAGE
    =================================================================== */
    if ( $( '#gd-community-app' ).length ) loadCommunity(1);

    function loadCommunity( page ) {
        post( 'growdiary_get_public', { page } ).done( function( res ) {
            if ( res.success ) {
                res.data.diaries.forEach( d => { diaryStore[ d.id ] = d; } );
                renderDiaryGrid( res.data.diaries, '#gd-community-grid', false );
                renderPagination( res.data, '#gd-community-pagination', loadCommunity );
            }
        } );
    }

    function renderPagination( data, selector, loadFn ) {
        if ( data.pages <= 1 ) { $( selector ).html(''); return; }
        let html = '';
        for ( let i = 1; i <= data.pages; i++ ) {
            html += `<button class="gd-btn gd-btn-sm ${i === data.current ? 'gd-btn-primary' : ''}" data-page="${i}">${i}</button>`;
        }
        $( selector ).html( html ).find('button').on( 'click', function() {
            loadFn( parseInt( $( this ).data('page'), 10 ) );
        } );
    }

    /* ===================================================================
       HELPERS
    =================================================================== */

    /**
     * formToObj – serialises a form, SKIPPING hidden inputs.
     * This is the core of the edit-diary fix: the form contains
     * hidden inputs (diary_id, week_number) that were being spread
     * into formData AFTER the explicit value we set, overwriting it.
     * Now we skip hidden inputs here and set those IDs explicitly.
     */
    function formToObj( $form ) {
        const obj = {};
        $form.find( 'input:not([type=hidden]), select, textarea' ).each( function() {
            const name = $( this ).attr('name');
            if ( name ) obj[ name ] = $( this ).val();
        } );
        return obj;
    }

    function escHtml( str ) {
        return String( str )
            .replace( /&/g, '&amp;' ).replace( /</g, '&lt;' )
            .replace( />/g, '&gt;' ).replace( /"/g, '&quot;' );
    }

} );
