<?php
/**
 * GrowDiary – AJAX Handlers
 * All endpoints require a valid nonce and logged-in user.
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/* -----------------------------------------------------------------------
   Helper: verify request
----------------------------------------------------------------------- */
function growdiary_verify() {
    if ( ! check_ajax_referer( 'growdiary_nonce', 'nonce', false ) ) {
        wp_send_json_error( [ 'message' => 'Invalid nonce.' ], 403 );
    }
    if ( ! is_user_logged_in() ) {
        wp_send_json_error( [ 'message' => 'You must be logged in.' ], 401 );
    }
    return get_current_user_id();
}

/* =======================================================================
   DIARIES
======================================================================= */

/* --- Create diary --- */
add_action( 'wp_ajax_growdiary_create_diary', 'growdiary_ajax_create_diary' );
function growdiary_ajax_create_diary() {
    global $wpdb;
    $user_id = growdiary_verify();

    $data = [
        'user_id'      => $user_id,
        'title'        => sanitize_text_field( $_POST['title'] ?? '' ),
        'strain'       => sanitize_text_field( $_POST['strain'] ?? '' ),
        'grow_medium'  => sanitize_text_field( $_POST['grow_medium'] ?? '' ),
        'light_type'   => sanitize_text_field( $_POST['light_type'] ?? '' ),
        'light_watts'  => intval( $_POST['light_watts'] ?? 0 ) ?: null,
        'grow_space'   => sanitize_text_field( $_POST['grow_space'] ?? '' ),
        'nutrients'    => sanitize_textarea_field( $_POST['nutrients'] ?? '' ),
        'start_date'   => sanitize_text_field( $_POST['start_date'] ?? '' ) ?: null,
        'cover_image'  => esc_url_raw( $_POST['cover_image'] ?? '' ),
        'visibility'   => in_array( $_POST['visibility'] ?? '', [ 'public', 'private' ] ) ? $_POST['visibility'] : 'private',
        'notes'        => sanitize_textarea_field( $_POST['notes'] ?? '' ),
    ];

    if ( empty( $data['title'] ) ) {
        wp_send_json_error( [ 'message' => 'Title is required.' ] );
    }

    $wpdb->insert( growdiary_table( 'diaries' ), $data );
    $diary_id = $wpdb->insert_id;

    if ( ! $diary_id ) {
        wp_send_json_error( [ 'message' => 'Could not save diary.' ] );
    }

    wp_send_json_success( [ 'diary_id' => $diary_id, 'message' => 'Grow diary created!' ] );
}

/* --- Read: list user's diaries --- */
add_action( 'wp_ajax_growdiary_get_diaries', 'growdiary_ajax_get_diaries' );
function growdiary_ajax_get_diaries() {
    global $wpdb;
    $user_id = growdiary_verify();

    $rows = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT d.*, 
             (SELECT COUNT(*) FROM " . growdiary_table('weeks') . " w WHERE w.diary_id = d.id) AS week_count
             FROM " . growdiary_table('diaries') . " d
             WHERE d.user_id = %d ORDER BY d.created_at DESC",
            $user_id
        ),
        ARRAY_A
    );

    wp_send_json_success( $rows );
}

/* --- Read: single diary --- */
add_action( 'wp_ajax_growdiary_get_diary', 'growdiary_ajax_get_diary' );
add_action( 'wp_ajax_nopriv_growdiary_get_diary', 'growdiary_ajax_get_diary' );
function growdiary_ajax_get_diary() {
    global $wpdb;
    $diary_id = intval( $_POST['diary_id'] ?? 0 );
    if ( ! $diary_id ) wp_send_json_error( [ 'message' => 'Invalid diary.' ] );

    $diary = $wpdb->get_row(
        $wpdb->prepare( "SELECT * FROM " . growdiary_table('diaries') . " WHERE id = %d", $diary_id ),
        ARRAY_A
    );

    if ( ! $diary ) wp_send_json_error( [ 'message' => 'Diary not found.' ] );

    // Private diaries: only owner can see
    if ( $diary['visibility'] === 'private' && $diary['user_id'] != get_current_user_id() ) {
        wp_send_json_error( [ 'message' => 'This diary is private.' ], 403 );
    }

    $weeks = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM " . growdiary_table('weeks') . " WHERE diary_id = %d ORDER BY week_number ASC",
            $diary_id
        ),
        ARRAY_A
    );

    // Decode JSON fields
    foreach ( $weeks as &$w ) {
        $decoded = json_decode( $w['images'] ?: '[]', true );
        $w['images'] = is_array( $decoded ) ? $decoded : [];
        $w['chart_data'] = ! empty( $w['chart_data'] ) ? json_decode( $w['chart_data'], true ) : null;
    }
    unset( $w ); // break reference

    $author = get_userdata( $diary['user_id'] );
    $diary['author_name'] = $author ? $author->display_name : 'Unknown';

    wp_send_json_success( [ 'diary' => $diary, 'weeks' => $weeks ] );
}

/* --- Update diary --- */
add_action( 'wp_ajax_growdiary_update_diary', 'growdiary_ajax_update_diary' );
function growdiary_ajax_update_diary() {
    global $wpdb;
    $user_id  = growdiary_verify();
    $diary_id = intval( $_POST['diary_id'] ?? 0 );

    $existing = $wpdb->get_row( $wpdb->prepare(
        "SELECT * FROM " . growdiary_table('diaries') . " WHERE id = %d AND user_id = %d",
        $diary_id, $user_id
    ) );
    if ( ! $existing ) wp_send_json_error( [ 'message' => 'Not found or not yours.' ] );

    $data = [
        'title'       => sanitize_text_field( $_POST['title'] ?? '' ),
        'strain'      => sanitize_text_field( $_POST['strain'] ?? '' ),
        'grow_medium' => sanitize_text_field( $_POST['grow_medium'] ?? '' ),
        'light_type'  => sanitize_text_field( $_POST['light_type'] ?? '' ),
        'light_watts' => intval( $_POST['light_watts'] ?? 0 ) ?: null,
        'grow_space'  => sanitize_text_field( $_POST['grow_space'] ?? '' ),
        'nutrients'   => sanitize_textarea_field( $_POST['nutrients'] ?? '' ),
        'start_date'  => sanitize_text_field( $_POST['start_date'] ?? '' ) ?: null,
        'cover_image' => esc_url_raw( $_POST['cover_image'] ?? '' ),
        'visibility'  => in_array( $_POST['visibility'] ?? '', [ 'public', 'private' ] ) ? $_POST['visibility'] : 'private',
        'status'      => in_array( $_POST['status'] ?? '', [ 'active','harvested','failed' ] ) ? $_POST['status'] : 'active',
        'notes'       => sanitize_textarea_field( $_POST['notes'] ?? '' ),
    ];

    $wpdb->update( growdiary_table('diaries'), $data, [ 'id' => $diary_id, 'user_id' => $user_id ] );
    wp_send_json_success( [ 'message' => 'Diary updated.' ] );
}

/* --- Delete diary --- */
add_action( 'wp_ajax_growdiary_delete_diary', 'growdiary_ajax_delete_diary' );
function growdiary_ajax_delete_diary() {
    global $wpdb;
    $user_id  = growdiary_verify();
    $diary_id = intval( $_POST['diary_id'] ?? 0 );

    $existing = $wpdb->get_row( $wpdb->prepare(
        "SELECT id FROM " . growdiary_table('diaries') . " WHERE id = %d AND user_id = %d",
        $diary_id, $user_id
    ) );
    if ( ! $existing ) wp_send_json_error( [ 'message' => 'Not found.' ] );

    $wpdb->delete( growdiary_table('weeks'),   [ 'diary_id' => $diary_id ] );
    $wpdb->delete( growdiary_table('diaries'), [ 'id'       => $diary_id ] );

    wp_send_json_success( [ 'message' => 'Diary deleted.' ] );
}

/* =======================================================================
   WEEKLY ENTRIES
======================================================================= */

/* --- Save (create or update) a weekly entry --- */
add_action( 'wp_ajax_growdiary_save_week', 'growdiary_ajax_save_week' );
function growdiary_ajax_save_week() {
    global $wpdb;
    $user_id     = growdiary_verify();
    $diary_id    = intval( $_POST['diary_id'] ?? 0 );
    $week_number = intval( $_POST['week_number'] ?? 0 );

    if ( $week_number < 1 || $week_number > 20 ) {
        wp_send_json_error( [ 'message' => 'Week must be 1–20.' ] );
    }

    // Ownership check
    $diary = $wpdb->get_row( $wpdb->prepare(
        "SELECT id FROM " . growdiary_table('diaries') . " WHERE id = %d AND user_id = %d",
        $diary_id, $user_id
    ) );
    if ( ! $diary ) wp_send_json_error( [ 'message' => 'Diary not found or not yours.' ] );

    $allowed_stages = [ 'germination','seedling','vegetative','flowering','flushing','harvest' ];
    $images_raw     = $_POST['images'] ?? '[]';
    $images         = json_decode( stripslashes( $images_raw ), true );
    if ( ! is_array( $images ) ) $images = [];
    $images = array_map( 'esc_url_raw', $images );

    $data = [
        'diary_id'     => $diary_id,
        'user_id'      => $user_id,
        'week_number'  => $week_number,
        'stage'        => in_array( $_POST['stage'] ?? '', $allowed_stages ) ? $_POST['stage'] : 'vegetative',
        'height_cm'    => is_numeric( $_POST['height_cm'] ?? '' )    ? floatval( $_POST['height_cm'] )    : null,
        'ph_water'     => is_numeric( $_POST['ph_water'] ?? '' )     ? floatval( $_POST['ph_water'] )     : null,
        'ph_runoff'    => is_numeric( $_POST['ph_runoff'] ?? '' )    ? floatval( $_POST['ph_runoff'] )    : null,
        'ec_ppm'       => is_numeric( $_POST['ec_ppm'] ?? '' )       ? intval( $_POST['ec_ppm'] )         : null,
        'temp_c'       => is_numeric( $_POST['temp_c'] ?? '' )       ? floatval( $_POST['temp_c'] )       : null,
        'humidity_pct' => is_numeric( $_POST['humidity_pct'] ?? '' ) ? intval( $_POST['humidity_pct'] )   : null,
        'vpd'          => is_numeric( $_POST['vpd'] ?? '' )          ? floatval( $_POST['vpd'] )          : null,
        'light_hours'  => is_numeric( $_POST['light_hours'] ?? '' )  ? intval( $_POST['light_hours'] )    : null,
        'watering_ml'  => is_numeric( $_POST['watering_ml'] ?? '' )  ? intval( $_POST['watering_ml'] )    : null,
        'nutrients'    => sanitize_textarea_field( $_POST['nutrients']    ?? '' ),
        'observations' => sanitize_textarea_field( $_POST['observations'] ?? '' ),
        'problems'     => sanitize_textarea_field( $_POST['problems']     ?? '' ),
        'images'       => json_encode( $images ),
        'rating'       => isset( $_POST['rating'] ) && is_numeric( $_POST['rating'] ) ? intval( $_POST['rating'] ) : null,
        'chart_data'   => ! empty( $_POST['chart_data'] ) ? wp_kses_post( stripslashes( $_POST['chart_data'] ) ) : null,
    ];

    // Upsert
    $existing = $wpdb->get_var( $wpdb->prepare(
        "SELECT id FROM " . growdiary_table('weeks') . " WHERE diary_id = %d AND week_number = %d",
        $diary_id, $week_number
    ) );

    if ( $existing ) {
        $wpdb->update( growdiary_table('weeks'), $data, [ 'id' => $existing ] );
    } else {
        $wpdb->insert( growdiary_table('weeks'), $data );
    }

    wp_send_json_success( [ 'message' => "Week {$week_number} saved." ] );
}

/* --- Delete a weekly entry --- */
add_action( 'wp_ajax_growdiary_delete_week', 'growdiary_ajax_delete_week' );
function growdiary_ajax_delete_week() {
    global $wpdb;
    $user_id     = growdiary_verify();
    $diary_id    = intval( $_POST['diary_id'] ?? 0 );
    $week_number = intval( $_POST['week_number'] ?? 0 );

    $diary = $wpdb->get_row( $wpdb->prepare(
        "SELECT id FROM " . growdiary_table('diaries') . " WHERE id = %d AND user_id = %d",
        $diary_id, $user_id
    ) );
    if ( ! $diary ) wp_send_json_error( [ 'message' => 'Diary not found.' ] );

    $wpdb->delete( growdiary_table('weeks'), [ 'diary_id' => $diary_id, 'week_number' => $week_number ] );
    wp_send_json_success( [ 'message' => "Week {$week_number} deleted." ] );
}

/* =======================================================================
   COMMUNITY: public diaries
======================================================================= */
add_action( 'wp_ajax_growdiary_get_public',        'growdiary_ajax_get_public' );
add_action( 'wp_ajax_nopriv_growdiary_get_public', 'growdiary_ajax_get_public' );
function growdiary_ajax_get_public() {
    global $wpdb;
    check_ajax_referer( 'growdiary_nonce', 'nonce' );

    $per_page = 12;
    $page     = max( 1, intval( $_POST['page'] ?? 1 ) );
    $offset   = ( $page - 1 ) * $per_page;

    $rows = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT d.*, u.display_name AS author_name,
             (SELECT COUNT(*) FROM " . growdiary_table('weeks') . " w WHERE w.diary_id = d.id) AS week_count
             FROM " . growdiary_table('diaries') . " d
             LEFT JOIN {$wpdb->users} u ON u.ID = d.user_id
             WHERE d.visibility = 'public'
             ORDER BY d.created_at DESC
             LIMIT %d OFFSET %d",
            $per_page, $offset
        ),
        ARRAY_A
    );

    $total = $wpdb->get_var(
        "SELECT COUNT(*) FROM " . growdiary_table('diaries') . " WHERE visibility = 'public'"
    );

    wp_send_json_success( [
        'diaries'    => $rows,
        'total'      => intval( $total ),
        'pages'      => ceil( $total / $per_page ),
        'current'    => $page,
    ] );
}
