<?php
/**
 * GrowDiary – Database layer
 * Creates two custom tables:
 *   growdiary_diaries  – one row per grow/diary
 *   growdiary_weeks    – up to 20 rows per diary (weekly log entries)
 */

if ( ! defined( 'ABSPATH' ) ) exit;

function growdiary_create_tables() {
    global $wpdb;
    $charset = $wpdb->get_charset_collate();

    $diaries_table = $wpdb->prefix . 'growdiary_diaries';
    $weeks_table   = $wpdb->prefix . 'growdiary_weeks';

    $sql = "
    CREATE TABLE IF NOT EXISTS {$diaries_table} (
        id            BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        user_id       BIGINT(20) UNSIGNED NOT NULL,
        title         VARCHAR(255)        NOT NULL DEFAULT '',
        strain        VARCHAR(255)        NOT NULL DEFAULT '',
        grow_medium   VARCHAR(100)        NOT NULL DEFAULT '',
        light_type    VARCHAR(100)        NOT NULL DEFAULT '',
        light_watts   SMALLINT UNSIGNED            DEFAULT NULL,
        grow_space    VARCHAR(100)        NOT NULL DEFAULT '',
        nutrients     TEXT,
        start_date    DATE                         DEFAULT NULL,
        cover_image   VARCHAR(500)                 DEFAULT '',
        visibility    ENUM('public','private')     DEFAULT 'private',
        status        ENUM('active','harvested','failed') DEFAULT 'active',
        notes         TEXT,
        created_at    DATETIME            NOT NULL DEFAULT CURRENT_TIMESTAMP,
        updated_at    DATETIME            NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY   (id),
        KEY user_id   (user_id)
    ) {$charset};

    CREATE TABLE IF NOT EXISTS {$weeks_table} (
        id            BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        diary_id      BIGINT(20) UNSIGNED NOT NULL,
        user_id       BIGINT(20) UNSIGNED NOT NULL,
        week_number   TINYINT UNSIGNED    NOT NULL DEFAULT 1,
        stage         ENUM('germination','seedling','vegetative','flowering','flushing','harvest') DEFAULT 'vegetative',
        height_cm     DECIMAL(6,2)                 DEFAULT NULL,
        ph_water      DECIMAL(4,2)                 DEFAULT NULL,
        ph_runoff     DECIMAL(4,2)                 DEFAULT NULL,
        ec_ppm        SMALLINT UNSIGNED            DEFAULT NULL,
        temp_c        DECIMAL(4,1)                 DEFAULT NULL,
        humidity_pct  TINYINT UNSIGNED             DEFAULT NULL,
        vpd           DECIMAL(4,2)                 DEFAULT NULL,
        light_hours   TINYINT UNSIGNED             DEFAULT NULL,
        watering_ml   SMALLINT UNSIGNED            DEFAULT NULL,
        nutrients     TEXT,
        observations  TEXT,
        problems      TEXT,
        images        TEXT,
        rating        TINYINT UNSIGNED             DEFAULT NULL,
        chart_data    LONGTEXT                     DEFAULT NULL,
        created_at    DATETIME            NOT NULL DEFAULT CURRENT_TIMESTAMP,
        updated_at    DATETIME            NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY   (id),
        UNIQUE KEY    diary_week (diary_id, week_number),
        KEY user_id   (user_id)
    ) {$charset};
    ";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta( $sql );
}

/* -----------------------------------------------------------------------
   Helper: get table names
----------------------------------------------------------------------- */
function growdiary_table( $name ) {
    global $wpdb;
    $tables = [
        'diaries' => $wpdb->prefix . 'growdiary_diaries',
        'weeks'   => $wpdb->prefix . 'growdiary_weeks',
    ];
    return $tables[ $name ] ?? '';
}

/* -----------------------------------------------------------------------
   Runtime upgrade: add columns to existing installs without re-activating
----------------------------------------------------------------------- */
function growdiary_maybe_upgrade() {
    global $wpdb;
    $table = growdiary_table( 'weeks' );
    $col   = $wpdb->get_results( "SHOW COLUMNS FROM `{$table}` LIKE 'chart_data'" );
    if ( empty( $col ) ) {
        $wpdb->query( "ALTER TABLE `{$table}` ADD COLUMN `chart_data` LONGTEXT DEFAULT NULL" );
    }
}
add_action( 'init', 'growdiary_maybe_upgrade' );
